package com.example.bedeveloper.config;

import com.example.bedeveloper.dto.jwt.TokenResponse;
import com.example.bedeveloper.entities.TokenEntity;
import com.example.bedeveloper.repository.TokenRepository;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.security.Keys;
import jakarta.annotation.PostConstruct;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.stereotype.Component;

import javax.crypto.SecretKey;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Objects;

@Component
public class JwtUtil {

    private SecretKey key;

    @Value("${jwt.secret}")
    private String jwtSecret;

    @Value("${jwt.expiration}")
    private int jwtExpirationMs;

    @Autowired
    private TokenRepository tokenRepository;

    @PostConstruct
    public void init() {
        this.key = Keys.hmacShaKeyFor(jwtSecret.getBytes(StandardCharsets.UTF_8));
    }


    public TokenResponse generateToken(String username, String role) {
        String jwt = Jwts.builder()
                .claim("username", username)
                .claim("role", role)
                .issuedAt(new Date())
                .expiration(new Date(System.currentTimeMillis() + jwtExpirationMs))
                .signWith(key)
                .compact();
        TokenEntity tokenEntity = new TokenEntity();
        tokenEntity.setToken(jwt);
        tokenEntity.setUsername(username);
        tokenRepository.save(tokenEntity);
        return new TokenResponse(jwt, username, new Date(System.currentTimeMillis() + jwtExpirationMs), role);
    }

    public String getUsername(String token) {
        if(Objects.isNull(token) || !validateToken(token)){
            return null;
        }
        return Jwts.parser()
                .verifyWith(key)
                .build()
                .parseSignedClaims(token)
                .getPayload().get("username").toString();
    }

    public boolean validateToken(String token) {
        TokenEntity tokenEntity = tokenRepository.findByToken(token);
        if(Objects.isNull(tokenEntity)){
            return false;
        }
        try {
            Jwts.parser().verifyWith(key).build().parseSignedClaims(token);
            return true;
        } catch (Exception e) {
            System.out.println(e.getMessage());
            return false;
        }
    }

    public String getTokenFromHeader(HttpHeaders headers) {
        String token = headers.getFirst("Authorization");
        if(Objects.isNull(token) || token.isEmpty()){
            return null;
        }
        if (token.startsWith("Bearer ")) {
            return token.substring(7);
        }
        return null;
    }
}
